const express = require("express");
const fs = require("fs");
const path = require("path");
const cors = require("cors");

const app = express();
const port = process.env.PORT || 3100;

// 📁 Ruta base de almacenamiento
const bucket = "/home/codemake/bucket";

// === 🟢 CORS universal y preflight ===
app.use(cors());
app.options("*", (req, res) => {
    res.header("Access-Control-Allow-Origin", "*");
    res.header("Access-Control-Allow-Methods", "GET,POST,OPTIONS");
    res.header("Access-Control-Allow-Headers", "Content-Type, x-filename");
    res.sendStatus(204);
});

// 🚀 Sin límites de tamaño ni timeout
app.use((req, res, next) => {
    req.setTimeout(0);
    res.setTimeout(0);
    next();
});

// 🏠 Ruta base
app.get("/", (req, res) => {
    res.header("Access-Control-Allow-Origin", "*");
    res.json({ message: "✅ Servidor de streaming activo y sin límites" });
});

// 📤 Carga directa de archivos (sin límite)
app.post("/upload-direct", (req, res) => {
    try {
        const ambiente = req.query.ambiente || "desa";
        const subfolder = req.query.subfolder || "final";
        const originalName = req.headers["x-filename"] || "archivo.bin";

        const destino = path.join(bucket, ambiente, subfolder);
        fs.mkdirSync(destino, { recursive: true });
        const filePath = path.join(destino, originalName);

        console.log(`📡 Iniciando streaming directo → ${filePath}`);
        const writeStream = fs.createWriteStream(filePath);

        let bytes = 0;
        let lastLog = Date.now();

        req.on("data", (chunk) => {
            bytes += chunk.length;
            const now = Date.now();
            if (now - lastLog >= 2000) {
                console.log(`📈 Progreso: ${(bytes / 1024 / 1024).toFixed(2)} MB escritos...`);
                lastLog = now;
            }
        });

        req.pipe(writeStream);

        req.on("end", () => {
            const mb = (bytes / 1024 / 1024).toFixed(2);
            console.log(`✅ Archivo recibido completamente (${mb} MB)`);
            res.header("Access-Control-Allow-Origin", "*");
            res.json({ ok: true, message: "Archivo subido correctamente", sizeMB: mb });
        });

        req.on("error", (err) => {
            console.error("❌ Error de transmisión:", err);
            res.header("Access-Control-Allow-Origin", "*");
            res.status(500).json({
                ok: false,
                error: "Error de transmisión (posible corte o límite del proxy)",
                detail: err.message,
            });
        });

        writeStream.on("error", (err) => {
            console.error("❌ Error al escribir archivo:", err);
            res.header("Access-Control-Allow-Origin", "*");
            res.status(500).json({
                ok: false,
                error: "Error al escribir archivo en disco",
                detail: err.message,
            });
            try {
                if (fs.existsSync(filePath)) fs.unlinkSync(filePath);
            } catch { }
        });
    } catch (err) {
        console.error("🔥 Error interno en /upload-direct:", err);
        res.header("Access-Control-Allow-Origin", "*");
        res.status(500).json({
            ok: false,
            error: "Error interno del servidor",
            detail: err.message,
        });
    }
});

// 🔥 Manejo global de errores y 404
app.use((err, req, res, next) => {
    console.error("🔥 Error global:", err);
    res.header("Access-Control-Allow-Origin", "*");
    if (err.type === "entity.too.large") {
        return res.status(413).json({
            ok: false,
            error: "Archivo demasiado grande para este endpoint (usa /upload-direct)",
        });
    }
    res.status(500).json({ ok: false, error: "Error interno", detail: err.message });
});

app.use((req, res) => {
    res.header("Access-Control-Allow-Origin", "*");
    res.status(404).json({ ok: false, error: "Ruta no encontrada" });
});

app.listen(port, () => {
    console.log(`🚀 Servidor de streaming sin límites escuchando en puerto ${port}`);
});
