// server.js (index.js)
const express = require('express');
const cors = require('cors');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const connection = require('./db');
const app = express();

app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

const PORT = process.env.PORT || 3000;
const SECRET_KEY = 'Test2024$';

app.get('/', async function (req, res) {
    return res.send({ mensaje: "conected to finances API" });
});


// Define the route for getting users
app.get('/users', (req, res) => {
    const query = 'SELECT id, user, email FROM users';

    connection.query(query, (err, results) => {
        if (err) {
            console.error('Error fetching users:', err);
            return res.status(500).json({ error: 'An error occurred while fetching users.' });
        }

        res.json(results);
    });
});


const validateEmail = (email) => {
    const re = /\S+@\S+\.\S+/;
    return re.test(email);
};


// Register user
app.post('/register', (req, res) => {

    const user = req.body.user || "";
    const password = req.body.password || "";
    const email = req.body.email || "";

    // Validate required fields
    if (user == "") return res.status(400).send({ error: 'user is required.' });
    if (password == "") return res.status(400).send({ error: 'password is required.' });
    if (email == "") return res.status(400).send({ error: 'email is required.' });
    if (!validateEmail(email)) return res.status(400).send({ error: 'invalid email format.' });


    // Check for duplicate user or email
    connection.query('SELECT * FROM users WHERE user = ? OR email = ?', [user, email], (err, results) => {
        if (err) return res.status(500).send('Error on the server.');

        if (results.length > 0) {
            return res.status(400).send({ message: 'User or email already exists.' });
        }

        // Hash the password before storing
        const hashedPassword = bcrypt.hashSync(password, 8);

        // Insert the new user into the database
        connection.query(
            'INSERT INTO users (user, password, email) VALUES (?, ?, ?)',
            [user, hashedPassword, email],
            (err, results) => {
                if (err) return res.status(500).send('Error on the server.');
                res.status(201).send({ message: 'User registered successfully!' });
            }
        );
    });
});


// Login and generate JWT
app.post('/login', (req, res) => {

    const user = req.body.user || "";
    const password = req.body.password || "";

    // Validate required fields
    if (user == "") return res.status(400).send({ error: 'user is required.' });
    if (password == "") return res.status(400).send({ error: 'password is required.' });

    connection.query('SELECT * FROM users WHERE user = ?', [user], (err, results) => {
        if (err) return res.status(500).send('Error on the server.');
        if (results.length === 0) return res.status(404).send('No user found.');

        const userRecord = results[0];

        // Compare passwords
        const passwordIsValid = bcrypt.compareSync(password, userRecord.password);
        if (!passwordIsValid) return res.status(401).send({ auth: false, token: null });

        const expiresIn = '5m';

        // Generate a token
        const token = jwt.sign({ id: userRecord.id, user: userRecord.user }, SECRET_KEY, { expiresIn });

        return res.status(200).send({ auth: true, expiresIn, token });
    });
});


const verifyToken = (req, res, next) => {
    // Get the Authorization header
    const authHeader = req.headers['authorization'];

    // Check if the header is in the expected format: "Bearer <token>"
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
        return res.status(401).json({ message: 'No token provided or incorrect format' });
    }

    // Extract the token from the header
    const token = authHeader.split(' ')[1];

    if (!token) return res.status(401).json({ message: 'No token provided' });

    // Verify the token
    jwt.verify(token, SECRET_KEY, (err, user) => {
        if (err) {
            // Check if the error is due to token expiration
            if (err.name === 'TokenExpiredError') {
                return res.status(401).json({ message: 'Token has expired' });
            }

            console.error('Token verification error:', err); // Log any other verification errors
            return res.status(403).json({ message: 'Invalid token' });
        }

        // Attach user data to the request object
        req.user = user;
        next(); // Proceed to the next middleware or route
    });
};


// CRUD for transactions (uses the imported connection)
app.post('/transactions', verifyToken, (req, res) => {

    try {
        insertRandomTransactions();
        return res.status(201).send({ message: "rows created" });
    }
    catch (err) {
        return res.status(500).send({ error: "error :" + err.message });
    }
});

app.get('/transactions', verifyToken, (req, res) => {
    connection.query('SELECT * FROM transactions', (err, results) => {
        if (err) throw err;
        return res.send(results);
    });
});



// Function to insert random records
const insertRandomTransactions = () => {
    for (let i = 0; i < 2; i++) {
        const description = `Transaction ${Math.floor(Math.random() * 1000)}`;
        const amount = (Math.random() * 100).toFixed(2); // Random amount between 0 and 100
        const type = Math.random() > 0.5 ? 'income' : 'expense'; // Randomly choose income or expense

        connection.query(
            'INSERT INTO transactions (description, amount, type) VALUES (?, ?, ?)',
            [description, amount, type],
            (err, results) => {
                if (err) throw err;
                console.log(`Inserted: ${description} - $${amount} - ${type}`);
            }
        );
    }
};

// const seconds = 10 * 1000;

// // Run the insert process every 30 seconds
// setInterval(() => {
//     console.log('Inserting 10 random transactions...');
//     insertRandomTransactions();
// }, seconds); // 30 seconds

app.listen(PORT, () => console.log('API running at http://localhost:' + PORT));
